*! version 0.8.0  02march2025
program define gmss, eclass sortpreserve
	version 19.0
	
	/*
		gmss.ado is a front-end to the GMSS mata library
		
	*/
	
    syntax anything [if] [in] [, *]				// gmss cmd name [#] , options
    quietly {
        gettoken cmd cmdargs : anything

		if "`cmd'" == substr("clear", 1, max(5, length("`cmd'"))) {
			Clear `cmdargs', `options'
			exit
        }
		if "`cmd'" == substr("display",  1, max(4, length("`cmd'")))  {
			noi Display `cmdargs', `options'
		}
		if "`cmd'" == substr("distribution",  1, max(4, length("`cmd'"))) {
			SetDist `cmdargs', `options'
		}
		if "`cmd'" == substr("link",  1, max(4, length("`cmd'"))) {
			SetLink `cmdargs', `options'
		}
		if "`cmd'" == substr("optimization",  1, max(3, length("`cmd'"))) {
			SetOpt `cmdargs', `options'
		}
		if "`cmd'" == substr("init",  1, max(4, length("`cmd'"))) {
			noi Init `cmdargs', `options'
		}
		if "`cmd'" == substr("run",  1, max(3, length("`cmd'"))) {
			noi Run `cmdargs' `if' `in', `options'
		}
		if "`cmd'" == substr("results",  1, max(3, length("`cmd'"))) {				// undocumented so far
			noi DisplayResults `cmdargs', `options'
		}
		return clear
    }
end


program define Clear			// Clear GMSS information out of dataset characteristics
	syntax [anything] [, *]
	
	/*
		Examples:
			Clear D1
			Clear *
			Clear D*
	*/
	
	if "`anything'" == "*" {						// Clear all GMSS-related characteristics
		local charnames : char _dta[]				// Load all characteristic names 
		foreach name in `charnames' {				// Loop over the characteristic names
			if substr("`name'", 1, 5) == "GMSS_" {  // Any characteristic that starts with GMSS_ is a match
				char _dta[`name'] ""
			}
		}
	}
	else {
		local match 0
		if substr("`anything'",length("`anything'"),1) == "*" {
			local anything = substr("`anything'", 1, length("`anything'")-1)
			local match 1
		}
		local charnames : char _dta[]
		local k = 10 + length("`anything'")
		foreach name in `charnames' {
			foreach abbr in Dist Link Optm {
				if `match' {
					if substr("`name'", 1, `k') == "GMSS_`abbr'_`anything'"  {
						char _dta[`name'] ""
					}
				}
				else {
					if "`name'" == "GMSS_`abbr'_`anything'"  {
						char _dta[`name'] ""
					}
				}
			}
		}
	}
end


program define Display	
	syntax anything [, noSKip colsize(integer 0)]
	
	/*
		Examples
		
		gmss display D1  
	*/
	
	local name `anything'
	if "`name'" == "*" {
		local charnames : char _dta[]					// Load all characteristic names 
		local n : word count `charnames'
		forvalues k=`n'(-1)1 {							// Loop over the characteristic names
			local cnam : word `k' of `charnames'
			if substr("`cnam'", 1, 10) == "GMSS_Dist_" {  // Any characteristic that starts with GMSS_ is a match
				local dist = substr("`cnam'", 11, .)
				noi DisplayName `dist' , `skip' colsize(`colsize')
			}
		}
		forvalues k=`n'(-1)1 {							// Loop over the characteristic names
			local cnam : word `k' of `charnames'
			if substr("`cnam'", 1, 10) == "GMSS_Optm_" {  // Any characteristic that starts with GMSS_ is a match
				local optm = substr("`cnam'", 11, .)
				noi DisplayName `optm' , `skip' colsize(`colsize')
			}
		}
	} 
	else {
		if "`_dta[GMSS_Dist_`name']'" != "" | "`_dta[GMSS_Optm_`name']'" != "" {
			noi DisplayName `name' , `skip' colsize(`colsize')
		}
	}
	if "`skip'" == "" {
		noi di 
	}
end
	
program define DisplayName
	syntax anything [, noSKip colsize(integer 0)]

	local name `anything'
	if "`name'" == "" {
		exit
	}
	
	if "`skip'" == "" {
		noi di 
	}
	if "`_dta[GMSS_Dist_`name']'" != "" {
		noi display as txt "Distribution: " _c 
		noi Format "GMSS(`name') `_dta[GMSS_Dist_`name']'", colsize(`colsize')
	}
	GetLinks `name' 0
	local link `r(link)'
	foreach lnk in `link' {
		if "`_dta[GMSS_Link_`lnk']'" != "" {
			noi display as txt "Link        : " _c 
			noi Format "GMSS(`lnk') `_dta[GMSS_Link_`lnk']'", colsize(`colsize')
		}
	}
	if "`_dta[GMSS_Optm_`name']'" != "" {
		noi display as txt "Optimization: " _c 
		noi Format "GMSS(`name') `_dta[GMSS_Optm_`name']'", colsize(`colsize')
	}
end

program define DisplayEName
	syntax anything [, noSKip colsize(integer 0) noOPT]

	local name `anything'
	if "`name'" == "" {
		exit
	}
	
	if "`skip'" == "" {
		noi di 
	}
	mata: st_local("str", `name'.darg)
	if "`str'" != "" {
		noi display as txt "Distribution: " _c 
		noi Format "`str'", colsize(`colsize')
	}
	GetLinks `name' 0
	local link `r(link)'
	local n
	forvalues k=1/`e(k_eq)' {
		mata: st_local("str", `name'.larg[`k'])
		if "`str'" != "" {
			noi display as txt "Link        : " _c 
			noi Format "`str'", colsize(`colsize')
		}
	}
	mata: st_local("str", `name'.oarg)
	if "`str'" != "" & "`opt'" == "" {
		noi display as txt "Optimization: " _c 
		noi Format "`str'", colsize(`colsize')
	}
    noi di as txt          "Version     : " as result "`e(version)'"
end

program define Format
	syntax anything [, colsize(integer 0)]
	
	/*
		Formats the display of GMSS strings to the current linesize.
		Shows all options with arguments as results (bold).
	*/
	
	local colstart = 15
	if `colsize' == 0 {
		local colsize = `c(linesize)'
	}    
	
	local size  = `colsize' - `colstart'
	local csize = 0
	tokenize `anything', parse("()")
	local k 1
	while "`1'" != "" {
		if `csize' + (`csize'!= 0) + length("`1'`2'`3'`4'") > `size' {
			noi di 
			local csize = 0
			noi di _col(15) _c 
		}
		if `csize' != 0 {
			noi di as txt " " _c
		}
		noi display as txt "`1'`2'" as result "`3'" as txt "`4'" _c
		local csize  = `csize' + (`csize'!= 0) + length("`1'`2'`3'`4'")
		mac shift 4
	}
	if `csize' != 0 {		// Add another line break if needed
		noi di 
	}
end

program define Init, rclass
	syntax [ anything ] [, noWARNing null OPTim(string) DIST(string)]
	
	quietly {
												// !! Could allow default: anything=GMSS
		local nn : word count `anything'
		if `nn' != 1 {
			noi di as err "gmss init needs one argument for the GMSS name"
			exit 198
		}
		local rname `anything'
		
		//tempvar touse
		//capture drop `touse'
		//gen byte `touse' = 1

        local ndist : word count `dist'
		
		local offset = 0
		if `ndist' == 3 {						
			local ddist : word 3 of `dist'
			GetName `ddist'
			local name3 `r(name)'	
			//markout `touse' `r(touse)'
			
			GetLinks `ddist' `offset'
			local link   `r(link)'  
			local lnames `r(names)' 
			local llabs  `r(labs)'   
			local vlist  `r(vlist)' 
			local offset : word count `link'
		}
		
		if `ndist' >= 2 {
			local hdist : word 2 of `dist'
			GetName `hdist'
			local name2 `r(name)'
			//markout `touse' `r(touse)'
			
			if "`name2'" != "heaped" & `ndist'==3 {
				noi di as err "Invalid specification of distributions"
				exit 199					
			}
			
			if "`name2'" == "heaped" {
				local hname `name2'			// Will add this heap() argument just before calling MATA
				local aname `name3'
			}
			else {
				local aname `name2'			// Will add this alt() argument just before calling MATA
			}
			
			GetLinks `hdist' `offset'
			local link   `link'   `r(link)' 
			local lnames `lnames' `r(names)' 
			local llabs  `llabs'  `r(labs)'  
			local vlist  `vlist' `r(vlist)'	
			local nl: word count `r(link)'
			if "`name2'" != "heaped" {
				local offset = `offset' + `nl'
			}
		}
		
		
		local bdist  : word 1 of `dist'		// Get distribution name(s)
		GetName `bdist'
		local name1 `r(name)'
		//markout `touse' `r(touse)'
		
		GetLinks `bdist'	`offset'			// Get links, linkarguments, varlists, etc
		local link   `link'   `r(link)'			// In-order names of link arguments corresponding to vector of links
		local lnames `lnames' `r(names)'		// In-order names of link arguments (filled in from GMSS_DistUtil.mata if needed)
		local llabs  `llabs'  `r(labs)'			// In-order names of link arguments (filled in from GMSS_DistUtil.mata if needed)
		local vlist  `vlist'  `r(vlist)'		// Get varlist (to be used to create touse variable and to manage null model)
		local offset : word count `r(link)'
		local link1 : word 1 of `link'
		
		local npar : word count `lnames'			// Total number of parameters
		local nparchk : word count `link'
		
		if `npar' != `nparchk' {
			noi di as err "Incorrect number of link functions for specified model"
			exit 198
		}
		
		//noi di as err "link = |`link'|  lnames=|`lnames'|  llabs=|`llabs'|"
		//noi di as err "vlist = |`vlist'|"
		
		//markout `touse' `vlist'					// Markout touse indicator (use same varlist even if running null model)
		//count if `touse'
		//local nobs `r(N)'						// Create number of obs argument
		
		//noi di as err "nobs = `nobs'"
		

		if "`aname'" != "" {
			local aarg alt(`aname')
		}
		if "`hname'" != "" {
			local harg heap(`hname')
		}

		// darg : Distribution argument for call to MATA 
		
		local opts name heap touse npar nobs alt 	// Allowed distribution options
		local defv xxx  xxx  xxx   					// Default values (xxx means that there is no default value)
	
		GetOptions `_dta[GMSS_Dist_`bdist']' `aarg' `harg' touse(`touse') npar(`npar') nobs(`nobs'), opts(`opts') defv(`defv') // Get the specified options
		local darg `r(options)'	// Set the data characteristic

		// larg: Build the vector of strings of Link arguments for call to MATA
		local larg (
		local comma
		forvalues k=1/`npar' {
			local lnk  : word `k' of `link'
			local name : word `k' of `lnames'
			local llab : word `k' of `llabs'
			
			//noi di as err "lnk(`lnk') name(`name') llab(`llab')"
			
			// Add link label (if not specified by user)
			local labarg label(`llab')
			if index("`_dta[GMSS_Link_`lnk']'", "label(") {
				local labarg
			}
			
			// Add link name (if not specified by user)
			local namarg name(`name')
			if index("`_dta[GMSS_Link_`lnk']'", "name(") {
				local namarg
			}
			
			local namelab `namarg' `labarg'
			
			if `k' == 1 & "`null'" != "" {		// Remove x() argument and ensure cons(on)
				RemoveXandCons `lnk'
				local larg `larg' `comma' `"`r(options)' `namelab'"'
			}
			else {
				local larg `larg' `comma' `"`_dta[GMSS_Link_`lnk']' `namelab'"'
			}
			local comma ,
		}
		local larg `larg' )
		
		//noi di as err `"larg = `larg'"'
		
		// oarg: Option argument for call to MATA !! add all default options here
		if "`optim'" == "" {
			gmss opt default
			local optim default
		}
		local oarg `_dta[GMSS_Optm_`optim']' 
	}
	
	// Last chance to bail out.  Check that link for zeroinflatedtau/ninflatedtau is constant-only; check that heap distribution links are all log
	if "`name1'" == "zeroinflatedtau" | "`name1'" == "ninflatedtau" | "`name1'" == "zeroninflatedtau"{
		local dist `_dta[GMSS_Dist_]'
		if (index("`_dta[GMSS_Link_`lnk']'", "x(") | index("`_dta[GMSS_Link_`lnk']'", "offset(") |  ///
			index("`_dta[GMSS_Link_`lnk']'", "exposure(") | index("`_dta[GMSS_Link_`lnk']'", "cons(off)")) {
			noi di as error "zeroinflatedtau/ninflatedtau/zeroninflatedtau must use a constant-only link (`lnk')"
			exit 197
		}
	}
	
	// !! Ensure that all links for "heaped" include an aval() that is unique and feasible
	// !! Ensure that all links for "multinomial" include an aval() that is unique and feasible (and completely cover the range of y); need one to be base
	
	// noi di as err `"darg = |`darg'|"'
	// noi di as err `"larg = |`larg'|"'
	// noi di as err `"oarg = |`oarg'|"'
	
	
	
	local vv = cond("`warning'"!="", "quietly", "noisily")
	

	//noi di as err "warning = |`warning'|  vv = |`vv'|"
	`vv' mata: GMSS_init("`darg'", `larg', "`oarg'", "`rname'", "`name1' `name2' `name3'")    // Creates mata object (struct GMSS) with global name `rname'
	
	// noi di as err "set vlist = |`vlist'|"
	qui mata: `rname'.vlist = "`vlist'"
		
	
	//!! Now, set the touse variable
	//noi di as err `"mata: moptimize_init_touse(`rname'.optm, "`touse'")"'
	//mata: moptimize_init_touse(`rname'.optm, "`touse'")
	// char _dta[GMSS_`rname'_title] `name1' `name2' `name3'
end


program define Run, eclass byable(recall)
	syntax [ anything ] [if] [in] [fweight pweight] [, noLOG noTABle noHEader noOPT title(string) ]
	
	/*
		Executes the optimization of the named distribution using the names applied
		in the -gmss dist- and -gmss link- statements (honoring the optimization options
		specified in the -gmss opt- statement)
		
		nolog		indicates that the value of the log-likelihood need not be displayed
		noheader	indicates that the header information need not be displayed
		notable		indicates that the regression table need not be displayed
		noopt		indicates that the optimization information need not be displayed
		title		indicates an optional title to print before optimization commences
		
		This routine sets up the large input for the mata optimization routine, but does so
		without changing any of the saved characteristics.
		
		gmss run Dist1
		gmss run 
		gmss run, noheader

		
		Handle [if] [in] and touse()
		
	*/

	local rname `anything'
	if "`rname'" == "" {
		local rname "GMSS"
	}

	if "`anything'" == "" {
		DisplayResults , `header' `table' `opt'
		exit
	}
	
	ereturn clear
	if "`title'" != "" & "`log'" == "" & "`header'"=="" {
		noi di _n as txt "`title'"
	}
	
	//!! Now, set the touse variable
	tempvar touse
	mark `touse' `if' `in' `weight'
	mata: st_local("vlist", `rname'.vlist)
	markout `touse' `vlist'
	
	preserve 
	noi cap noi {
		qui keep if `touse'==1
		
		// noi di "vlist = |`vlist'|"
		qui count if `touse'
		mata: `rname'.dist->nobs = `r(N)'


		
		//noi di as err `"mata: moptimize_init_touse(`rname'.optm, "`touse'")"'
		//mata: moptimize_init_touse(`rname'.optm, "`touse'")
		
		local vv = cond("`log'"!="", "quietly", "noisily")
		`vv' mata: GMSS_optimize(`rname', "rc")
		if `rc' {
			exit `rc'
		}
		mata: GMSS_result_post(`rname')	
		local k = e(k_eq)
		tempname b V
		matrix `b' = e(b)
		matrix `V' = e(V)
		capture ereturn post `b' `V'

		mata: st_numscalar("e(k_eq)"      , `k')
		mata: st_numscalar("e(converged)" , moptimize_result_converged(`rname'.optm))
		mata: st_numscalar("e(niter)"     , moptimize_result_iterations(`rname'.optm))
		mata: st_numscalar("e(ll0)"       , moptimize_result_value0(`rname'.optm))
		mata: st_numscalar("e(ll)"        , moptimize_result_value(`rname'.optm))
		
		eret local predict   GMSS_pred
		eret local estat_cmd GMSS_estat
		eret local mataname `rname'
		eret local moptname `rname'.optm
		eret local cmd      gmss
		//eret local GMSS     `rname'
		
		if 0 {										// !! unsure if want to do this
			// Save separate segments of beta
			forvalues k=`e(k_eq)'(-1)1 {
				mata: st_matrix("e(b`k')", moptimize_result_eq_coefs(`rname'.optm,`k'))
			}
		}
		
		// Save available variance matrices
		// mata: st_matrix("e(V_opg)", moptimize_result_V_opg(`rname'.optm))
		// mata: st_matrix("e(V_oim)", moptimize_result_V_oim(`rname'.optm))
		// mata: st_matrix("e(V_robust)", moptimize_result_V_oim(`rname'.optm))
		
		// Save vcetype and title
		mata: st_global("e(vcetype)"     , moptimize_result_Vtype(`rname'.optm))
		mata: st_global("e(title)"       , `rname'.title)
		mata: st_global("e(version)", `rname'.gmssver)
		
		if "`optim'" == "" {
			local optim default
		}

		// local oarg `_dta[GMSS_Optm_`optim''
		mata: st_local("oarg", `rname'.oarg)
		tokenize `oarg', parse("()")
		while "`1'" != "" {
			capture confirm number `3'
			if _rc == 0 {
				eret scalar `1' = `3'
			}
			else {
				eret local `1' `3'
			}
			mac shift 4
		}
		
		DisplayResults `rname', `header' `table' `opt'
	}
	restore
end


program define DisplayResults, eclass
	syntax [anything] [, noHEader noTABle noOPT ]
	
	local vv1 = cond("`header'"!="", "quietly", "noisily")
	local vv2 = cond("`table'"!="", "quietly", "noisily")

	`vv1' {
		if "`header'" == "" {
			DisplayHeader , title(`e(title)') 
		}
		DisplayEName `anything', `opt' colsize(80) noskip
	}
	
	`vv2' ereturn display
end

program define DisplayHeader
/*
      Source |       SS           df       MS      Number of obs   =       250
12345678901234567890123456789012345678901234567890123456789012345678901234567890
         1         2         3         4         5         6         7         8
Generalized Model Specification System(GMSS)
*/
	syntax [ anything ] [, TItle(string)]
	noi di 
	noi di as txt "Gen. Model Specification System" _c
	noi di as txt %47s strproper("`title' regression")  
end

program define RemoveXandCons, rclass
	args name
	
	quietly {
		local opts name label ancillary n    y    offset exposure init aval		// Allowed link options
		local defv xxx  xxx   xxx       xxx  xxx  xxx    xxx      xxx  xxx		// Default values (xxx means there there is no default value)
	
		GetOptions `_dta[GMSS_Link_`name']', opts(`opts') defv(`defv')	null // Get the specified options
		ret local options `r(options)'
	}
end

program define GetName, rclass
	args name
	
	quietly {
		local opts name heap touse link				// Allowed distribution options
		local defv xxx  xxx  xxx   xxx				// Default values (xxx means that there is no default value)
	
		GetOptions `_dta[GMSS_Dist_`name']', opts(`opts') defv(`defv') // Get the specified options
		ret local name `r(name)'
		ret local touse `r(touse)'
	}
end

program define GetNameLabel, rclass
	args dname link k offset
	quietly {
		local opts name label ancillary n    y    x    cons offset exposure init aval	// Allowed link options
		local defv xxx  xxx   xxx       xxx  xxx  xxx  on   xxx    xxx      xxx  xxx	// Default values (xxx means there there is no default value)
		
		local kpo = `k' 
		if "`dname'"=="heaped" {
			local kpo = `k' + `offset'
		}
		
		local terms mu sigma tau delta 
		local lab : word `kpo' of `terms'
		if "`lab'" == "" {
			local lab par`k'
		}
		if index("zeroinflatedzeroinflatedtauzeromarginalzeroalteredzeromarginaltau", "`dname'") {
			local lab P0
		}
		if index("ninflatednalterednmarginalninflatedtau", "`dname'") {
			local lab Pn
		}
		
		GetOptions `_dta[GMSS_Link_`link']', opts(`opts') defv(`defv') // Get the specified options
		local name  `r(name)'
		local labl  `r(label)'
		local vlist `r(varlist)'
		local aval  `r(aval)'
		
		if "`name'" == "" {
			mata: `dname'_link(`k', "name") 
		}
		if "`labl'" == "" {
			if "`dname'"=="heaped" {
				local labl modulo_`aval'
			}
			else {
				local labl `name'_`lab'
			}
		}

		ret local name  `name'
		ret local label `labl'
		ret local vlist `vlist'
	}
end

program define GetLinks, rclass
	args name offset
	
	quietly {
		local opts name heap touse link 				// Allowed distribution options
		local defv xxx  xxx  xxx   xxx					// Default values (xxx means that there is no default value)
	
		GetOptions `_dta[GMSS_Dist_`name']', opts(`opts') defv(`defv') // Get the specified options
		local dname `r(name)'
		local lnks `r(link)'
		
		local i = 1
		foreach lnk in `lnks' {
			GetNameLabel `dname' `lnk' `i' `offset'
			local lnames `lnames' `r(name)'
			local llabs  `llabs'  `r(label)'
			local vlist  `vlist'  `r(vlist)'
			local ++i
		}
		
		ret local link  `lnks'
		ret local names `lnames'
		ret local labs  `llabs'
		ret local vlist `vlist'
	}
end

program define SetDist, rclass						// Define a data characteristic for a distribution
	syntax anything [, *] 
	
	if length("`anything'") > 20 {
		noi di as err "GMSS specified name is too long: `anything'"
		exit 198
	}
	
	local opts name heap alt touse link					// Allowed distribution options
	local defv xxx  xxx  xxx xxx   xxx					// Default values (xxx means that there is no default value)
	
	GetOptions `options', opts(`opts') defv(`defv') // Get the specified options
	char _dta[GMSS_Dist_`anything'] `r(options)'	// Set the data characteristic
	
	// noi di as err "char(`anything')=|`r(options)'|   all from |`options'|"
	
	local roptions `r(options)'
	local rtouse   `r(touse)'
	local rlink    `r(link)'
	local rname    `r(name)'
	
	// noi di as err "rtouse = |`rtouse'|"
	
	mata: `rname'_npar("nchk")
	local npar : word count `rlink'
	
	if `npar' != `nchk' & "`rname'" != "heaped" & "`rname'" != "ordered" & "`rname'"!="ordered2" {
		noi di as err "Incorrect number of link objects for `rname'"
		exit 198
	}
	
	// Split the information into useful individual r() macros
	ret local options `roptions'					
	ret local touse   `rtouse'
	ret local link    `rlink'
	ret local name    `rname'
end

program define SetLink, rclass
	syntax anything [, *] 
	
	if length("`anything'") > 20 {
		noi di as err "GMSS specified name is too long: `anything'"
		exit 198
	}
	
	local opts name label ancillary n    y    x    cons offset exposure init aval	// Allowed link options
	local defv xxx  xxx   xxx       xxx  xxx  xxx  on   xxx    xxx      xxx  xxx	// Default values (xxx means there there is no default value)
	
	GetOptions `options', opts(`opts') defv(`defv')	// Get the specified options
	char _dta[GMSS_Link_`anything'] `r(options)'	// Set the data characteristic

	// Split the information into useful individual r() macros
	ret local options `r(options)'
	ret local varlist `r(varlist)'
	ret local name    `r(name)'
	ret local label   `r(label)'
end

program define SetOpt, rclass
	syntax anything [, *] 
	
	local opts evaluator technique cluster by   tracelevel tracevalue maxiter svy ptol vtol nrtol warning ignorenrtol search random singular valueid        oim
	local defv d2        nr        xxx     xxx  value      on         300     off 1e-6 1e-7 1e-5  on      off         on     off    hybrid   Log-likelihood on
	
	GetOptions `options', opts(`opts') defv(`defv')
	char _dta[GMSS_Optm_`anything'] `r(options)'	// Set the data characteristic
	ret local options `r(options)'
end

program define GetOptions, rclass							// Process options for distributions, links, and optimizer
	syntax [anything] , opts(string) defv(string) [ null ]
	
	/*
		Handles a collection of options `anything'.  Will only process those options
		listed in `opts' with default values given in `defv'.
		
		Ignores options specified in `anything' that are not part of `opts'
		
		All options have the syntax:  option_name(option_value)
	*/
	
	local nnnn : word count `opts'		// Grab number of supported options to process
	
	/*
		Examples of anything:
			name(zeroinflated) touse(touse) links(L1)
			x(x1 z1 w1) name(log) aval(6) label(modulo_6)
			evaluator(d2debug) search(on) maxiter(15) tracelevel(hessian)
			
			every argument should be 4 parts:
			         1      2      3     4
				option_name ( option_val )
			or it might be 3 parts if the option_val is not specified
			         1      2  3    
				option_name (  )
			
			
			
			NOTE:  you can name an option more than once.  If you do, the latest version applies
			NOTE:  unknown options generate an error
			NOTE:  none of the options allow abbreviation
	*/
	tokenize "`anything'", parse("()") 
	while "`1'" != "" {
		if "`2'" != "(" {
			noi di as err "syntax error in specification of options"
			exit 198
		}
		
		// See if you can find the specified option in the list of supported options
		local flag 0
		forvalues k=1/`nnnn' {
			local optval : word `k' of `opts'
			if "`1'" == "`optval'" {
				local flag 1
			}
		}
		
		// If the option was not in the list of supported options, but it is 
		// either the x() or the cons(on/off) option, that is ok so long as 
		// the null option is specified (meaning that a null model is desired)
		if `flag' == 0 {
			if ("`optval'" == "x" | "`optval'"=="cons") & "`null'" == "" {
				noi di as err "unknown option (`1')"
				exit 198
			}
		}
		
		// Check if option_val is missing and then skip 
		if "`3'" == ")" {
			local `1'       // Set the macro option to null (will be set to default later)
			mac shift 3
		}
		else {
			// Make sure that all 4 parts are present
			if "`4'" != ")" {
				noi di as err "syntax error in specification of options"
				exit 198
			}
			local `1' `3'	// Set the macro option_name to the option_value
			mac shift 4
		}
	}
	
	// Blank out return macros that build a full list of options (combining defaults with specifications)
	local lst
	local vl 

	// Loop over the allowed options again knowing that each option_name is 
	// either empty (so we can assign the default value), or has been specified.
	forvalues k=1/`nnnn' {
		local opt : word `k' of `opts'	// Grab the option_name
		local val : word `k' of `defv'	// Grab the default value for option_name
		if "`val'" == "xxx" {			// Erase default value if it is "xxx"
			local val
		}
		if "``opt''" == "" {			// If the option_name macro is empty, assign the default
			local `opt' `val'
		}
		
		// Handle on/off options and make sure only "on" or "off" is specified
		if "`val'" == "on" | "`val'" == "off" {
			if "``opt''" != "on" & "``opt''" != "off" {
				noi di as err "`opt'() can only be set to on or off"
				exit 198
			}
		}
		// Handle numeric options and ensure a real number has been specified
		capture confirm number `val'
		if _rc == 0 {
			capture confirm number ``opt''
			if _rc {
				noi di as err "`opt'() must be a real number"
				exit 198
			}
		}
		// Handle integer options and ensure an integer has been specified
		capture confirm integer number `val'
		if _rc == 0 {
			capture confirm integer number ``opt''
			if _rc {
				noi di as err "`opt'() must be an integer number"
				exit 198
			}
		}
		// Save touse variable for separate return (remove from characteristic specification)
		if "`opt'" != "touse" {
			if !missing("``opt''") {
				local lst `lst' `opt'(``opt'')
			}
		}
		else {
			local tt ``opt''
		}
		
		// Handle varlist related options so that complete varlist can be compiled to set the sample
		if "`opt'" == "y" | "`opt'" == "x" | "`opt'" == "offset" | "`opt'" == "exposure" {
			local vl `vl' ``opt''
		}
		// Grab name so that name argument can be returned separately
		if "`opt'" == "name" {
			local nm ``opt''
		}
		// Grab label so that label argument can be returned separately
		if "`opt'" == "label" {
			local labl ``opt''
		}
		// Grab links so that links argument can be returned separately
		if "`opt'" == "link" {
			local lnk ``opt''
		}
		// Grab links so that links argument can be returned separately
		if "`opt'" == "aval" {
			local aval ``opt''
		}
	}
	return local options `lst'
	return local varlist `vl'
	return local touse   `tt'
	return local name    `nm'
	return local label   `labl'
	return local link    `lnk'
	return local aval    `aval'
end


mata
void function GMSS_Display(string scalar optname)
{
	pointer(struct GMSS scalar) scalar gmssp

	gmssp = findexternal(optname)
	moptimize_result_display(gmssp->optm)
}

end


